<?php
// activity.php - Script CRON para Gerenciar logs de atividade e limpar conexões live inativas.

// Desativa limite de tempo e erros de exibição
set_time_limit(0); 
ini_set('display_errors', 0);
error_reporting(0);

// Inclui o arquivo de configuração e conexão com o banco de dados
// Certifique-se de que o caminho para 'config.php' está correto.
require_once 'config.php';

// ====================================================================
// CONFIGURAÇÃO DE TEMPOS
// ====================================================================

// O tempo (em segundos) que uma linha pode ficar na lines_live sem ser removida.
// 600 segundos = 10 minutos. Se o cliente não enviar um 'heartbeat' por 10 min, ele é considerado inativo.
$inactivity_timeout = 600; 

// O tempo (em segundos) que os logs de atividade devem ser mantidos. 
// 2592000 segundos = 30 dias.
$max_log_age = 2592000; 

// Obtém o timestamp atual
$current_time = time(); 

// ====================================================================
// ETAPA 1: Limpar Conexões Inativas (Mover de lines_live para lines_activity)
// ====================================================================

// 1. Encontra registros na lines_live que estão ativos há mais tempo que o timeout
// Nota: Seu lines_live não tem 'hls_last_read', então usamos 'date_start' como fallback
$sql_select_inactives = "
    SELECT 
        activity_id, user_id, stream_id, server_id, proxy_id, user_agent, user_ip, 
        container, date_start, geoip_country_code, isp, external_device, divergence, 
        hls_last_read, hls_end, fingerprinting, mac_id 
    FROM lines_live 
    WHERE date_start < (:current_time - :inactivity_timeout)";

$stmt = $pdo->prepare($sql_select_inactives);
$stmt->execute([
    ':current_time' => $current_time,
    ':inactivity_timeout' => $inactivity_timeout
]);
$inactive_connections = $stmt->fetchAll(PDO::FETCH_ASSOC);

$moved_count = 0;

if (!empty($inactive_connections)) {
    
    $pdo->beginTransaction();

    try {
        // 2. Insere os registros inativos na tabela de histórico (lines_activity)
        $sql_insert_activity = "
            INSERT INTO lines_activity (
                activity_id, user_id, stream_id, server_id, proxy_id, user_agent, user_ip, 
                container, date_start, date_end, geoip_country_code, isp, external_device, divergence
            ) VALUES (
                :activity_id, :user_id, :stream_id, :server_id, :proxy_id, :user_agent, :user_ip, 
                :container, :date_start, :date_end, :geoip_country_code, :isp, :external_device, :divergence
            )";
        
        $stmt_insert = $pdo->prepare($sql_insert_activity);
        
        $ids_to_delete = [];

        foreach ($inactive_connections as $conn) {
            // O date_end será o tempo atual
            $date_end = $current_time; 
            
            // Tenta executar a inserção no histórico
            $stmt_insert->execute([
                ':activity_id' => $conn['activity_id'],
                ':user_id' => $conn['user_id'],
                ':stream_id' => $conn['stream_id'],
                ':server_id' => $conn['server_id'],
                ':proxy_id' => $conn['proxy_id'],
                ':user_agent' => $conn['user_agent'],
                ':user_ip' => $conn['user_ip'],
                ':container' => $conn['container'],
                ':date_start' => $conn['date_start'],
                ':date_end' => $date_end,
                ':geoip_country_code' => $conn['geoip_country_code'],
                ':isp' => $conn['isp'],
                ':external_device' => $conn['external_device'],
                ':divergence' => $conn['divergence']
            ]);
            
            $ids_to_delete[] = $conn['activity_id'];
            $moved_count++;
        }
        
        // 3. Remove os registros que foram movidos da tabela lines_live
        if (!empty($ids_to_delete)) {
            $ids_string = implode(',', array_map('intval', $ids_to_delete));
            $pdo->exec("DELETE FROM lines_live WHERE activity_id IN ($ids_string)");
        }
        
        $pdo->commit();

    } catch (PDOException $e) {
        $pdo->rollBack();
        // Em um ambiente de produção, registre $e->getMessage() em um arquivo de log
        // echo "Erro ao mover conexões: " . $e->getMessage();
    }
}


// ====================================================================
// ETAPA 2: Limpar Logs Antigos (lines_activity)
// ====================================================================

$purge_time = $current_time - $max_log_age;

$sql_delete_old_logs = "DELETE FROM lines_activity WHERE date_start < :purge_time";

$stmt_delete = $pdo->prepare($sql_delete_old_logs);
$stmt_delete->execute([':purge_time' => $purge_time]);

$deleted_count = $stmt_delete->rowCount();


// ====================================================================
// FIM E REGISTRO DE EXECUÇÃO (Opcional)
// ====================================================================

// Se você precisar de um feedback de console (se rodar via linha de comando):
// echo "Cronjob activity.php executado.\n";
// echo "Conexões inativas movidas para logs: $moved_count\n";
// echo "Logs antigos deletados: $deleted_count\n";

?>
