<?php
// editar_bouquet.php - VERSÃO FINAL CORRIGIDA (Busca Híbrida de IDs)
// Resolve o problema de categorias que não salvam por causa de ["ID"] vs ID

require 'header.php'; 

// Aumenta tempo e memória para listas gigantes
set_time_limit(0);
ini_set('memory_limit', '-1');

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$pacote = [
    'bouquet_name' => '', 
    'bouquet_channels' => '[]', 
    'bouquet_movies' => '[]', 
    'bouquet_series' => '[]'
];

// --- 1. SALVAR DADOS ---
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $nome = $_POST['nome'];
    
    // Recebe os IDs via JSON (do Javascript)
    $cats_live   = json_decode($_POST['json_live'], true) ?? [];
    $cats_movie  = json_decode($_POST['json_movie'], true) ?? [];
    $cats_series = json_decode($_POST['json_series'], true) ?? [];

    $include_uncategorized_movies = in_array('0', $cats_movie);
    $cats_movie = array_diff($cats_movie, ['0']);

    // --- FUNÇÃO PODEROSA PARA BUSCAR CANAIS ---
    // Busca tanto pelo ID puro (12) quanto pelo ID formatado (["12"])
    function buscarIdsPorCategoria($pdo, $tabela, $lista_ids_cat, $tipo_stream = null) {
        if (empty($lista_ids_cat)) return [];
        
        $ids_encontrados = [];
        
        // Divide em lotes de 500 para não travar o banco
        $chunks = array_chunk($lista_ids_cat, 500);
        
        foreach ($chunks as $lote) {
            // 1. Monta lista de IDs puros: 12, 13, 14
            $lista_pura = implode(',', array_map('intval', $lote));
            
            // 2. Monta lista de IDs formatados: "12", "13", "14" (para busca REGEX ou LIKE)
            // A forma mais segura e rápida é usar REGEX para pegar ["12"]
            // Mas para compatibilidade máxima, vamos usar o método OR
            
            $sql = "";
            if ($tabela == 'streams') {
                // Busca ID puro OU ID com aspas/colchetes
                // category_id = 12 OR category_id LIKE '%"12"%'
                // Nota: Fazer isso para muitos IDs é lento, então usamos a estratégia de IN para puros
                // e REGEXP para os complexos se necessário.
                
                // Estratégia Otimizada: Buscar TUDO que bate com IN e depois filtrar? Não.
                // Vamos buscar onde o category_id contenha o número.
                
                // TRUQUE DO REPLACE: Removemos [ " ] do banco na hora de comparar
                // REPLACE(REPLACE(REPLACE(category_id, '[', ''), ']', ''), '"', '') IN (12, 13...)
                
                $sql = "SELECT id FROM streams WHERE type = $tipo_stream AND 
                        REPLACE(REPLACE(REPLACE(category_id, '[', ''), ']', ''), '\"', '') IN ($lista_pura)";
            } else {
                // Para Séries
                $sql = "SELECT id FROM streams_series WHERE 
                        REPLACE(REPLACE(REPLACE(category_id, '[', ''), ']', ''), '\"', '') IN ($lista_pura)";
            }
            
            $res = $pdo->query($sql)->fetchAll(PDO::FETCH_COLUMN);
            if ($res) {
                $ids_encontrados = array_merge($ids_encontrados, $res);
            }
        }
        
        return array_unique($ids_encontrados);
    }

    // A. CANAIS (LIVE)
    $ids_channels = buscarIdsPorCategoria($pdo, 'streams', $cats_live, 1);
    $count_live = count($ids_channels);
    $json_channels = $count_live > 0 ? json_encode(array_values(array_map('intval', $ids_channels))) : "[]";

    // B. FILMES (MOVIES)
    $ids_movies = buscarIdsPorCategoria($pdo, 'streams', $cats_movie, 2);
    
    // Filmes sem categoria
    if ($include_uncategorized_movies) {
        $sql_orphans = "SELECT id FROM streams WHERE type = 2 AND (category_id = 0 OR category_id IS NULL OR category_id = '[]' OR category_id = '')";
        $ids_orphans = $pdo->query($sql_orphans)->fetchAll(PDO::FETCH_COLUMN);
        if($ids_orphans) $ids_movies = array_merge($ids_movies, $ids_orphans);
    }
    
    $ids_movies = array_unique($ids_movies);
    $count_movie = count($ids_movies);
    $json_movies = $count_movie > 0 ? json_encode(array_values(array_map('intval', $ids_movies))) : "[]";

    // C. SÉRIES
    $ids_series = buscarIdsPorCategoria($pdo, 'streams_series', $cats_series);
    $count_series = count($ids_series);
    $json_series = $count_series > 0 ? json_encode(array_values(array_map('intval', $ids_series))) : "[]";

    try {
        if ($id > 0) {
            $stmt = $pdo->prepare("UPDATE bouquets SET bouquet_name=?, bouquet_channels=?, bouquet_movies=?, bouquet_series=? WHERE id=?");
            $stmt->execute([$nome, $json_channels, $json_movies, $json_series, $id]);
        } else {
            $stmt = $pdo->prepare("INSERT INTO bouquets (bouquet_name, bouquet_channels, bouquet_movies, bouquet_series) VALUES (?, ?, ?, ?)");
            $stmt->execute([$nome, $json_channels, $json_movies, $json_series]);
            $id = $pdo->lastInsertId(); 
        }

        $msg = "Pacote Salvo com Sucesso!\\n\\nAgora o SBT deve aparecer.\\nResumo:\\n- Canais: $count_live\\n- Filmes: $count_movie\\n- Séries: $count_series";
        echo "<script>alert(`$msg`); window.location='editar_bouquet.php?id=$id';</script>";
        exit;

    } catch (Exception $e) {
        echo "<div class='alert alert-danger'>Erro SQL: " . $e->getMessage() . "</div>";
    }
}

// --- 2. CARREGAR DADOS ---
$cats_checked_live = [];
$cats_checked_movie = [];
$cats_checked_series = [];
$check_orphans = false;

if ($id > 0) {
    $stmt = $pdo->prepare("SELECT * FROM bouquets WHERE id = ?");
    $stmt->execute([$id]);
    $pacote = $stmt->fetch(PDO::FETCH_ASSOC);

    // Função Otimizada para Recuperar (A mesma lógica reversa)
    function recuperarCategorias($pdo, $ids_json, $tabela, $is_orphan_check = false) {
        $ids = json_decode($ids_json, true);
        if (empty($ids)) return $is_orphan_check ? false : [];

        $cats_encontradas = [];
        $has_orphans = false;
        $chunks = array_chunk($ids, 2000); // Lotes maiores para leitura

        foreach ($chunks as $chunk) {
            $ids_str = implode(',', $chunk);
            if (!$is_orphan_check) {
                // Aqui pegamos o category_id cru do banco
                $sql = ($tabela == 'streams_series') 
                     ? "SELECT DISTINCT category_id FROM streams_series WHERE id IN ($ids_str)"
                     : "SELECT DISTINCT category_id FROM streams WHERE id IN ($ids_str)";
                
                $res = $pdo->query($sql)->fetchAll(PDO::FETCH_COLUMN);
                
                // Limpa os IDs que vêm como ["12"] para virar apenas 12
                foreach($res as $r) {
                    $limpo = str_replace(['[', ']', '"'], '', $r);
                    if(is_numeric($limpo)) $cats_encontradas[] = $limpo;
                }
            }
            if ($is_orphan_check) {
                $sql_orp = "SELECT count(*) FROM streams WHERE id IN ($ids_str) AND (category_id = 0 OR category_id IS NULL)";
                if ($pdo->query($sql_orp)->fetchColumn() > 0) $has_orphans = true;
            }
        }
        return $is_orphan_check ? $has_orphans : array_unique($cats_encontradas);
    }

    $cats_checked_live = recuperarCategorias($pdo, $pacote['bouquet_channels'], 'streams');
    $cats_checked_movie = recuperarCategorias($pdo, $pacote['bouquet_movies'], 'streams');
    $check_orphans = recuperarCategorias($pdo, $pacote['bouquet_movies'], 'streams', true);
    $cats_checked_series = recuperarCategorias($pdo, $pacote['bouquet_series'], 'streams_series');
}

// --- 3. LISTAR CATEGORIAS ---
$lista_cat_live = $pdo->query("SELECT id, category_name FROM streams_categories WHERE category_type = 'live' ORDER BY category_name ASC")->fetchAll(PDO::FETCH_ASSOC);
$lista_cat_movie = $pdo->query("SELECT id, category_name FROM streams_categories WHERE category_type = 'movie' ORDER BY category_name ASC")->fetchAll(PDO::FETCH_ASSOC);
$lista_cat_series = $pdo->query("SELECT id, category_name FROM streams_categories WHERE category_type = 'series' ORDER BY category_name ASC")->fetchAll(PDO::FETCH_ASSOC);
$count_orphans = $pdo->query("SELECT count(*) FROM streams WHERE type = 2 AND (category_id = 0 OR category_id IS NULL)")->fetchColumn();
?>

<div class="container mt-4 mb-5">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            
            <form method="POST" id="formBouquet">
                <input type="hidden" name="json_live" id="json_live">
                <input type="hidden" name="json_movie" id="json_movie">
                <input type="hidden" name="json_series" id="json_series">

                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3 class="fw-bold text-dark">
                        <?= $id > 0 ? 'Editar Pacote' : 'Criar Novo Pacote' ?>
                    </h3>
                    <button type="button" onclick="salvarComSeguranca()" class="btn btn-success btn-lg shadow">
                        <i class="fas fa-save me-2"></i> Salvar e Aplicar
                    </button>
                </div>

                <div class="card shadow-sm border-0 mb-4">
                    <div class="card-body">
                        <label class="form-label fw-bold">Nome do Pacote</label>
                        <input type="text" name="nome" class="form-control form-control-lg" required value="<?= htmlspecialchars($pacote['bouquet_name']) ?>">
                    </div>
                </div>

                <div class="card shadow-sm border-0">
                    <div class="card-header bg-white border-bottom-0 pt-3">
                        <ul class="nav nav-tabs card-header-tabs" id="myTab" role="tablist">
                            <li class="nav-item"><button class="nav-link active fw-bold" id="live-tab" data-bs-toggle="tab" data-bs-target="#live" type="button"><i class="fas fa-tv me-2 text-primary"></i>Canais TV</button></li>
                            <li class="nav-item"><button class="nav-link fw-bold" id="movies-tab" data-bs-toggle="tab" data-bs-target="#movies" type="button"><i class="fas fa-film me-2 text-warning"></i>Filmes</button></li>
                            <li class="nav-item"><button class="nav-link fw-bold" id="series-tab" data-bs-toggle="tab" data-bs-target="#series" type="button"><i class="fas fa-layer-group me-2" style="color: #6f42c1;"></i>Séries</button></li>
                        </ul>
                    </div>
                    
                    <div class="card-body p-0">
                        <div class="tab-content" id="myTabContent">
                            
                            <div class="tab-pane fade show active" id="live">
                                <div class="p-3 bg-light border-bottom d-flex justify-content-between">
                                    <span class="fw-bold text-primary">Categorias de TV</span>
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="marcarTodos('live')">Marcar Tudo</button>
                                </div>
                                <div class="p-4" style="max-height: 500px; overflow-y: auto;">
                                    <div class="row">
                                        <?php foreach($lista_cat_live as $c): 
                                            $checked = in_array($c['id'], $cats_checked_live) ? 'checked' : '';
                                        ?>
                                            <div class="col-md-6 col-lg-4 mb-2">
                                                <div class="form-check">
                                                    <input class="form-check-input check-live" type="checkbox" value="<?= $c['id'] ?>" id="live_<?= $c['id'] ?>" <?= $checked ?>>
                                                    <label class="form-check-label w-100 text-truncate" for="live_<?= $c['id'] ?>"><?= htmlspecialchars($c['category_name']) ?></label>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="movies">
                                <div class="p-3 bg-light border-bottom d-flex justify-content-between">
                                    <span class="fw-bold text-warning">Categorias de Filmes</span>
                                    <button type="button" class="btn btn-sm btn-outline-warning" onclick="marcarTodos('movie')">Marcar Tudo</button>
                                </div>
                                <div class="p-4" style="max-height: 500px; overflow-y: auto;">
                                    <div class="row">
                                        <?php if($count_orphans > 0): ?>
                                            <div class="col-12 mb-3">
                                                <div class="card border-warning bg-warning-subtle">
                                                    <div class="card-body d-flex align-items-center">
                                                        <div class="form-check">
                                                            <input class="form-check-input check-movie" type="checkbox" value="0" id="movie_0" style="transform: scale(1.3);" <?= $check_orphans ? 'checked' : '' ?>>
                                                            <label class="form-check-label fw-bold ms-2 text-dark" for="movie_0">
                                                                ⚠️ INCLUIR FILMES SEM CATEGORIA (<?= $count_orphans ?>)
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endif; ?>

                                        <?php foreach($lista_cat_movie as $c): 
                                            $checked = in_array($c['id'], $cats_checked_movie) ? 'checked' : '';
                                        ?>
                                            <div class="col-md-6 col-lg-4 mb-2">
                                                <div class="form-check">
                                                    <input class="form-check-input check-movie" type="checkbox" value="<?= $c['id'] ?>" id="movie_<?= $c['id'] ?>" <?= $checked ?>>
                                                    <label class="form-check-label w-100 text-truncate" for="movie_<?= $c['id'] ?>"><?= htmlspecialchars($c['category_name']) ?></label>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="series">
                                <div class="p-3 bg-light border-bottom d-flex justify-content-between">
                                    <span class="fw-bold" style="color: #6f42c1;">Categorias de Séries</span>
                                    <button type="button" class="btn btn-sm btn-outline-secondary" onclick="marcarTodos('series')">Marcar Tudo</button>
                                </div>
                                <div class="p-4" style="max-height: 500px; overflow-y: auto;">
                                    <div class="row">
                                        <?php foreach($lista_cat_series as $c): 
                                            $checked = in_array($c['id'], $cats_checked_series) ? 'checked' : '';
                                        ?>
                                            <div class="col-md-6 col-lg-4 mb-2">
                                                <div class="form-check">
                                                    <input class="form-check-input check-series" type="checkbox" value="<?= $c['id'] ?>" id="series_<?= $c['id'] ?>" <?= $checked ?>>
                                                    <label class="form-check-label w-100 text-truncate" for="series_<?= $c['id'] ?>"><?= htmlspecialchars($c['category_name']) ?></label>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
    function marcarTodos(tipo) {
        let checkboxes = document.querySelectorAll('.check-' + tipo);
        if(checkboxes.length === 0) return;
        let estado = !checkboxes[0].checked;
        checkboxes.forEach(c => c.checked = estado);
    }

    function salvarComSeguranca() {
        const coletarIDs = (classe) => {
            let ids = [];
            document.querySelectorAll('.' + classe + ':checked').forEach(el => {
                ids.push(el.value);
            });
            return JSON.stringify(ids);
        };
        document.getElementById('json_live').value = coletarIDs('check-live');
        document.getElementById('json_movie').value = coletarIDs('check-movie');
        document.getElementById('json_series').value = coletarIDs('check-series');
        document.getElementById('formBouquet').submit();
    }
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<div class="content">
