<?php
// server_monitor.php - Script CRON de API para receber estatísticas do Load Balancer
// Deve ser executado pelo cronjob do Load Balancer (Ex: a cada 5 minutos)

require_once 'config.php';
ini_set('display_errors', 0);
set_time_limit(5); // Este script deve ser rápido

// ====================================================================
// 1. CHAVE DE SEGURANÇA (CRÍTICO!)
// ====================================================================

// SUBSTITUA ESTA CHAVE PELA CHAVE QUE VOCÊ DEFINIU NO setup_loadbalancer.sh
$EXPECTED_SECRET_KEY = "Deus e fiel"; 

// ====================================================================
// 2. COLETA E VALIDAÇÃO DE DADOS
// ====================================================================

$secret_key = $_POST['secret_key'] ?? '';
$server_ip = $_POST['ip'] ?? '';
$cpu_load = (float)($_POST['cpu_load'] ?? 0);
$status = (int)($_POST['status'] ?? 0); // Status 1 = Online

// Validação Básica
if ($secret_key !== $EXPECTED_SECRET_KEY || empty($server_ip)) {
    // Retorna 403 Forbidden para o cronjob
    header('HTTP/1.1 403 Forbidden');
    die("Acesso Negado ou Chave Incorreta.");
}

// ====================================================================
// 3. BUSCAR ID DO SERVIDOR E ATUALIZAR STATUS
// ====================================================================

// Busca o ID do servidor na tabela 'servers' pelo IP que enviou o dado.
$stmt = $pdo->prepare("SELECT id FROM servers WHERE server_ip = ? LIMIT 1");
$stmt->execute([$server_ip]);
$server_data = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$server_data) {
    // Se o servidor não for encontrado, podemos ignorar ou registrar um erro.
    // O servidor deve ter sido adicionado manualmente na área "Gerenciar Servidores".
    die("Servidor com IP $server_ip não encontrado no Painel Master.");
}

$server_id = $server_data['id'];
$timestamp = time();

// Atualiza o status do servidor na tabela principal (servers)
try {
    $sql_update_server = "UPDATE servers SET status = ?, last_check_ago = ? WHERE id = ?";
    $pdo->prepare($sql_update_server)->execute([$status, $timestamp, $server_id]);
} catch (PDOException $e) {
    // Logar erro de atualização se necessário
}

// ====================================================================
// 4. ATUALIZAR/INSERIR ESTATÍSTICAS (servers_stats)
// ====================================================================

// O método mais simples é verificar se o servidor já tem uma entrada na servers_stats
// e atualizá-la, ou inserir uma nova. Usaremos ON DUPLICATE KEY UPDATE se o ID for chave.
// Assumindo que o 'server_id' é UNIQUE ou PRIMARY KEY na servers_stats:

try {
    $sql_insert_stats = "
        INSERT INTO servers_stats (server_id, connections, streams, users, cpu, timestamp)
        VALUES (?, 0, 0, 0, ?, ?)
        ON DUPLICATE KEY UPDATE 
            cpu = VALUES(cpu),
            timestamp = VALUES(timestamp),
            connections = connections, -- Mantém o valor atual (ou zera, dependendo da sua lógica)
            streams = streams,
            users = users
    ";
    
    $pdo->prepare($sql_insert_stats)->execute([$server_id, $cpu_load, $timestamp]);

} catch (PDOException $e) {
    // Se o INSERT falhar, tente um UPDATE simples (se a estrutura da tabela for diferente)
    try {
        $sql_update_stats = "
            UPDATE servers_stats SET 
                cpu = ?, 
                timestamp = ?
            WHERE server_id = ?
        ";
        $pdo->prepare($sql_update_stats)->execute([$cpu_load, $timestamp, $server_id]);
    } catch (PDOException $e) {
        // Falha final, registre se necessário.
        die("Falha ao atualizar a tabela servers_stats.");
    }
}

// Retorno de sucesso (o cronjob não receberá nada além de um código 200 OK)
die("OK");
?>
