<?php
// ======================================================================
//      SISTEMA DE ATUALIZAÇÃO DE EPG v2.3
//      Versão Final com renomeação de funções para evitar conflitos.
// ======================================================================
session_start();

// --- LÓGICA DE ATUALIZAÇÃO (EXECUTADA VIA AJAX) ---
if (isset($_GET['start_epg'])) {
    
    set_time_limit(0);
    ignore_user_abort(true);
    ini_set('memory_limit', '512M');
    require_once("./api/controles/db.php");
    
    $epg_url = "http://epg.topiptv.tv.br/epg.xml";

    // ======================================================================
    //      FUNÇÕES RENOMEADAS PARA EVITAR CONFLITO
    // ======================================================================
    function epg_normalize_name($name) {
        $name = strtolower($name);
        $name = str_replace(['hd', 'fhd', 'sd', '4k', '(br)', '(pt)'], '', $name);
        $name = preg_replace('/[^a-z0-9]/', '', $name);
        return $name;
    }
    
    function epg_send_message($type, $data) {
        echo "event: " . $type . "\n";
        echo "data: " . json_encode($data) . "\n\n";
        if (ob_get_level() > 0) {
            ob_flush();
        }
        flush();
    }
    // ======================================================================

    header('Content-Type: text/event-stream');
    header('Cache-Control: no-cache');
    
    try {
        $conexao = conectar_bd();
        epg_send_message('log', 'Baixando arquivo EPG...');
        $epg_content = @file_get_contents($epg_url);
        if ($epg_content === false) { throw new Exception("Falha ao baixar o arquivo EPG de: $epg_url"); }
        if (substr($epg_url, -3) == '.gz') { $epg_content = gzdecode($epg_content); }
        epg_send_message('log', '[OK] Download concluído.');

        epg_send_message('log', 'Analisando o arquivo XML...');
        $xml = new SimpleXMLElement($epg_content);
        $epg_data = [];
        foreach ($xml->channel as $channel) {
            $epg_data[epg_normalize_name((string)$channel->{'display-name'})] = (string)$channel['id'];
        }
        epg_send_message('log', '[OK] Análise concluída. '. count($epg_data) ." canais encontrados no arquivo EPG.");

        epg_send_message('log', 'Buscando seus canais no banco de dados...');
        $stmt = $conexao->query("SELECT id, name, epg_channel_id FROM streams WHERE stream_type = 'live'");
        $canais_do_banco = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $total_canais = count($canais_do_banco);
        epg_send_message('log', "[OK] Encontrados $total_canais canais no seu painel.");
        epg_send_message('stats', ['total_canais' => $total_canais]);

        epg_send_message('log', "---------------------------------\nIniciando comparação e atualização...");
        
        $encontrados = 0;
        $atualizados = 0;
        $processados = 0;

        foreach ($canais_do_banco as $canal) {
            $processados++;
            $nome_normalizado_banco = epg_normalize_name($canal['name']);
            
            if (isset($epg_data[$nome_normalizado_banco])) {
                $encontrados++;
                $epg_id = $epg_data[$nome_normalizado_banco];
                if ($canal['epg_channel_id'] !== $epg_id) {
                    $update_stmt = $conexao->prepare("UPDATE streams SET epg_channel_id = ? WHERE id = ?");
                    $update_stmt->execute([$epg_id, $canal['id']]);
                    if ($update_stmt->rowCount() > 0) {
                        $atualizados++;
                        epg_send_message('log', "-> [ATUALIZADO] '{$canal['name']}' -> EPG ID: $epg_id");
                    }
                }
            }
            
            $progresso = ($processados / ($total_canais ?: 1)) * 100;
            epg_send_message('progress', ['percent' => round($progresso), 'text' => "$processados/$total_canais"]);
            epg_send_message('stats', ['canais_atualizados' => $atualizados]);
        }

        $taxa_sucesso = ($total_canais > 0) ? round(($encontrados / $total_canais) * 100) : 0;
        $ignorados = $total_canais - $encontrados;
        
        $resumo = "Resumo: Total: $total_canais | Processados: $processados | Encontrados: $encontrados | Atualizados: $atualizados | Ignorados: $ignorados";
        epg_send_message('log', "---------------------------------\nPROCESSO CONCLUÍDO!");
        epg_send_message('done', [ 'status' => 'Atualizado com sucesso!', 'horario' => date('d/m/Y H:i:s'), 'resumo_footer' => $resumo, 'taxa_sucesso' => $taxa_sucesso ]);
    } catch (Exception $e) {
        epg_send_message('log', "\n--- OCORREU UM ERRO FATAL ---\n" . $e->getMessage());
        epg_send_message('done', ['status' => 'Falha na atualização', 'horario' => date('d/m/Y H:i:s')]);
    }
    exit();
}

// --- PÁGINA HTML ---
require_once("menu.php");
require_once("./api/controles/db.php");
$conexao = conectar_bd();
$total_canais = $conexao->query("SELECT COUNT(id) FROM streams WHERE stream_type = 'live'")->fetchColumn();
?>

<style>
    .stat-card { background-color: var(--bg-card); border: 1px solid var(--border-color); border-radius: 8px; padding: 20px; text-align: center; }
    .stat-card h3 { font-size: 1.2rem; color: var(--text-secondary); margin-bottom: 10px; font-weight: 600; text-transform: uppercase; }
    .stat-card p { font-size: 2.5rem; font-weight: 700; color: var(--text-primary); margin: 0; }
    .status-bar { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
    [data-theme="dark"] .status-bar { background-color: #198754; color: #fff; border-color: #198754; }
    .status-bar.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
    [data-theme="dark"] .status-bar.error { background-color: #dc3545; color: #fff; border-color: #dc3545; }
    .log-container { background-color: #212529; color: #f8f9fa; font-family: monospace;
                     padding: 20px; border-radius: 5px; height: 300px; overflow-y: scroll; 
                     white-space: pre-wrap; font-size: 14px; margin-top: 20px; }
</style>

<h4 class="mb-4 text-muted text-uppercase">Sistema de EPG</h4>

<div class="card mb-3">
    <div class="card-body">
        <div class="row status-bar p-2 rounded align-items-center" id="status-bar">
            <div class="col-md-4"><strong>Status da atualização:</strong> <span id="status-text">Aguardando início</span></div>
            <div class="col-md-4"><strong>Horário da atualização:</strong> <span id="status-horario">--/--/---- --:--:--</span></div>
            <div class="col-md-4"><strong>Atualizado por:</strong> <span id="status-por"><?php echo htmlspecialchars($_SESSION['username']); ?></span></div>
        </div>
    </div>
</div>

<div class="card mb-3">
    <div class="card-body">
        <div class="progress" style="height: 30px;">
            <div id="progress-bar" class="progress-bar progress-bar-striped bg-success" role="progressbar" style="width: 0%;">Progresso: 0/<?php echo $total_canais; ?> (0%)</div>
        </div>
    </div>
</div>

<div class="row mb-3 g-3">
    <div class="col-md-4">
        <div class="stat-card">
            <h3>Total de Canais</h3>
            <p id="total-canais"><?php echo $total_canais; ?></p>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card">
            <h3>Canais Atualizados</h3>
            <p id="canais-atualizados">0</p>
        </div>
    </div>
    <div class="col-md-4">
        <div class="stat-card">
            <h3>Uso de Memória</h3>
            <p id="uso-memoria">N/A</p>
        </div>
    </div>
</div>

<div class="text-center mb-3">
    <button id="startBtn" class="btn btn-success btn-lg">Iniciar Atualização do EPG</button>
</div>

<div class="card">
    <div class="card-body">
        <div class="form-check">
            <input class="form-check-input" type="checkbox" id="taxa-sucesso-check" disabled>
            <label class="form-check-label" for="taxa-sucesso-check" id="taxa-sucesso-label">Taxa de sucesso: 0%</label>
        </div>
        <div id="resumo-footer" class="mt-2 text-muted">
            Resumo: Total: <?php echo $total_canais; ?> | Processados: 0 | Encontrados: 0 | Atualizados: 0 | Ignorados: 0
        </div>
    </div>
</div>

<div class="log-container" id="log-box">Aguardando início do processo...</div>

<script>
$(document).ready(function() {
    const startBtn = $('#startBtn');
    const logBox = $('#log-box');
    const progressBar = $('#progress-bar');
    const statusBar = $('#status-bar');
    
    startBtn.on('click', function() {
        startBtn.prop('disabled', true).text('Atualizando...');
        logBox.html('Iniciando conexão com o servidor...\n');
        $('.progress-bar').addClass('progress-bar-animated');
        
        const eventSource = new EventSource('atualizar_epg.php?start_epg=true');

        eventSource.addEventListener('log', function(e) {
            logBox.append(JSON.parse(e.data) + '\n');
            logBox.scrollTop(logBox[0].scrollHeight);
        });

        eventSource.addEventListener('progress', function(e) {
            const data = JSON.parse(e.data);
            progressBar.css('width', data.percent + '%');
            progressBar.text('Progresso: ' + data.text + ' (' + data.percent + '%)');
        });

        eventSource.addEventListener('stats', function(e) {
            const data = JSON.parse(e.data);
            if(data.total_canais) $('#total-canais').text(data.total_canais);
            if(data.canais_atualizados >= 0) $('#canais-atualizados').text(data.canais_atualizados);
        });

        eventSource.addEventListener('done', function(e) {
            const data = JSON.parse(e.data);
            $('#status-text').text(data.status);
            $('#status-horario').text(data.horario);
            if(data.resumo-footer) $('#resumo-footer').text(data.resumo_footer);
            if(data.taxa_sucesso >= 0) {
                $('#taxa-sucesso-label').text('Taxa de sucesso: ' + data.taxa_sucesso + '%');
                $('#taxa-sucesso-check').prop('checked', true);
            }
            if (data.status.includes('Falha')) {
                statusBar.removeClass('bg-success').addClass('bg-danger');
            } else {
                statusBar.removeClass('bg-danger').addClass('bg-success');
            }
            
            startBtn.prop('disabled', false).text('Atualização Concluída, Recarregar Página');
            startBtn.off('click').on('click', function(){ location.reload(); });
            $('.progress-bar').removeClass('progress-bar-animated');
            
            eventSource.close();
        });

        eventSource.onerror = function() {
            logBox.append('\n--- ERRO ---\nConexão com o servidor perdida. O processo pode continuar em segundo plano. Tente recarregar a página em alguns minutos.');
            startBtn.prop('disabled', false).text('Erro! Tentar Novamente');
            eventSource.close();
        };
    });
});
</script>