/**
 * Script para gerenciar a página de categorias, incluindo a funcionalidade
 * de reordenação com drag-and-drop (arrastar e soltar) usando DataTables.
 */
$(document).ready(function() {

    // --- 1. INICIALIZAÇÃO DO DATATABLES ---
    // A tabela é inicializada com as configurações para reordenação.
    var table = $('#data_table').DataTable({
        // Habilita o processamento do lado do servidor para lidar com grandes volumes de dados.
        "processing": true,
        "serverSide": true,

        // Define a fonte dos dados da tabela.
        // IMPORTANTE: Deve apontar para o seu script de API que lista as categorias.
        // O script PHP deve retornar um JSON no formato que o DataTables espera.
        "ajax": {
            "url": "api/categorias.php", // Usando o endpoint da API que criamos.
            "type": "POST",
            "data": {
                "action": "listar" // Ação para buscar os dados.
            }
        },

        // Define as colunas e os dados que elas exibirão.
        "columns": [
            // A primeira coluna é para o ID e também servirá como "handle" para arrastar.
            // A classe 'reorder' é usada pelo JavaScript para identificar a área de arraste.
            { "data": "id_categoria", "className": "reorder text-center" },
            { "data": "category_name" },
            { "data": "tipo", "className": "text-center" },
            { "data": "is_adult", "className": "text-center" },
            { "data": "bg_ssiptv", "className": "text-center" },
            { "data": "acoes", "className": "text-center", "orderable": false }
        ],

        // Define a ordenação inicial da tabela.
        // Ordena pela primeira coluna (que deve ser a coluna 'ordem' ou 'id') em ordem ascendente.
        "order": [[0, 'asc']],

        // Tradução da interface do DataTables para Português-Brasil.
        "language": {
            "url": "./js/datatables/pt_br.json" // Verifique se este caminho está correto.
        },

        // --- 2. CONFIGURAÇÃO DO ROWREORDER ---
        // Esta é a parte principal que habilita a funcionalidade de arrastar e soltar.
        rowReorder: {
            selector: 'td.reorder', // Define que apenas a coluna com a classe 'reorder' pode ser usada para arrastar.
            dataSrc: 'id_categoria', // A coluna do JSON que será usada para identificar a linha.
            update: false // Impede que o DataTables envie os dados automaticamente. Faremos isso no nosso botão.
        }
    });

    // --- 3. EVENTO DE REORDENAÇÃO ---
    // Este evento é disparado sempre que o usuário solta uma linha em uma nova posição.
    table.on('row-reorder', function(e, diff, edit) {
        // Mostra e habilita o botão "Salvar Ordem" para que o usuário possa confirmar a alteração.
        $('#saveOrderBtn').prop('disabled', false).fadeIn();
    });

    // --- 4. EVENTO DO BOTÃO SALVAR ---
    // Define o que acontece quando o usuário clica no botão para salvar a nova ordem.
    $('#saveOrderBtn').on('click', function() {
        const btn = $(this);
        // Desabilita o botão e mostra um ícone de carregamento para dar feedback ao usuário.
        btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin me-1"></i> Salvando...');

        // Pega todos os dados da tabela na nova ordem usando a API do DataTables.
        // Este método é mais seguro e eficiente do que ler o HTML da tabela.
        var data = table.rows().data().toArray();

        // Mapeia os dados para um formato simples que a API espera: {id: id_da_categoria, ordem: nova_posicao}
        var orderData = data.map(function(row, index) {
            return {
                // IMPORTANTE: 'id_categoria' deve ser o nome exato da chave do ID no seu JSON.
                id: row.id_categoria,
                ordem: index + 1 // A nova ordem é simplesmente a posição da linha na tabela.
            };
        });

        // Envia os dados da nova ordem para a API via AJAX.
        $.ajax({
            url: 'api/categorias.php', // O endpoint da API.
            type: 'POST',
            data: {
                action: 'save_order', // A ação que o PHP irá processar.
                order: JSON.stringify(orderData) // Envia o array como uma string JSON.
            },
            success: function(response) {
                // Tenta interpretar a resposta do servidor como JSON.
                try {
                    var res = JSON.parse(response);
                    if (res.status === 'success') {
                        Swal.fire('Sucesso!', 'A ordem das categorias foi salva.', 'success');
                        // Esconde o botão novamente após salvar.
                        btn.fadeOut();
                        // Recarrega a tabela para garantir que a ordem está correta.
                        table.ajax.reload(null, false);
                    } else {
                        // Se a API retornar um erro conhecido.
                        throw new Error(res.message || 'Erro retornado pela API.');
                    }
                } catch (e) {
                    // Se a resposta não for um JSON válido ou houver outro erro.
                    Swal.fire('Erro!', 'Não foi possível salvar a ordem. Detalhe: ' + e.message, 'error');
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                // Se houver um erro de conexão ou um erro 500 no servidor.
                Swal.fire('Erro de Conexão!', 'Não foi possível se comunicar com o servidor: ' + errorThrown, 'error');
            },
            complete: function() {
                // Garante que o botão seja reativado e o texto restaurado, mesmo se houver erro.
                btn.prop('disabled', false).html('<i class="fas fa-save me-1"></i> Salvar Ordem');
            }
        });
    });
});
